;==================================================

;JEEAhk1FCGui

;==================================================

;AHK v2 GUI/Menu functions and classes for AHK v1 by jeeswg
;[first released: 2018-03-05]
;[updated: 2023-05-03]
;based on AHK v2.0.2
;tested with AHK v1.1.36

;==================================================

;KEY NOTES:

;known issues/limitations:
;Ctrl ContextMenu event, IsRightClick is always 1, even if Shift+10 or AppsKey (Menu key) was pressed
;Ctrl OnCommand and OnNotify events, these have not been extensively tested

;==================================================

;LINKS:

;e.g. test with:
;control zoo (AHK v2) - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=37&t=41685&p=197521#p197521

;see also:
;objects: backport AHK v2 Gui/Menu classes to AHK v1 - Page 2 - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=6&t=43530&p=204331#p204331
;commands as functions (AHK v2 functions for AHK v1) - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=37&t=29689

;==================================================

;NOT AVAILABLE / RELATED FUNCTIONS:

;not available: writable variables
;A_AllowMainWindow
;A_IconHidden
;A_IconTip

;TraySetIcon(FileName:="", IconNumber:="", Freeze:="")
;{
;	Menu, % "Tray", % "Icon", % FileName, % IconNumber, % Freeze
;}

;==================================================

;USE BYREF PARAMETERS:

;note: AHK v2 uses & for ByRef parameters, AHK v1 and these classes, don't

;oGui.GetClientPos()
;oGui.GetPos()
;oGuiCtrl.GetPos()

;==================================================

;NOTES ON TYPES:

;note: these backport classes don't give the correct type for Type()

;oCtl := oGui.Add("DropDownList") ;oCtl.Type returns DDL, Type(oCtl) returns Gui.DDL
;oCtl := oGui.Add("Picture") ;oCtl.Type returns Pic, Type(oCtl) returns Gui.Pic
;oCtl := oGui.Add("Tab2") ;oCtl.Type returns Tab2, Type(oCtl) returns Gui.Tab
;oCtl := oGui.Add("Tab3") ;oCtl.Type returns Tab3, Type(oCtl) returns Gui.Tab

;==================================================

;LINKS:

;LINKS - GENERAL
;Changes from v1.1 to v2.0
;https://autohotkey.com/v2/v2-changes.htm
;Thoughts for v2.0
;https://autohotkey.com/v2/v2-thoughts.htm
;list of every object type/property/method - AutoHotkey Community
;https://autohotkey.com/boards/viewtopic.php?f=37&t=44081

;LINKS - MENU (AHK V2)
;Menu/MenuBar Object - Methods & Properties | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/Menu.htm
;MenuFromHandle - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/MenuFromHandle.htm
;Variables and Expressions - Definition & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/Variables.htm#TrayMenu

;LINKS - GUI (AHK V2)
;Gui Object - Methods & Properties | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/Gui.htm
;GuiCtrlFromHwnd - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/GuiCtrlFromHwnd.htm
;GuiFromHwnd - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/GuiFromHwnd.htm
;OnEvent (GUI) - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/GuiOnEvent.htm

;LINKS - GUI CONTROLS (AHK V2)
;GuiControl Object - Methods & Properties | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/GuiControl.htm
;GUI Control Types - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/GuiControls.htm
;ListView (GUI) - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/ListView.htm
;TreeView (GUI) - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/TreeView.htm

;LINKS - MENU (AHK V1)
;Menu
;https://autohotkey.com/docs/commands/Menu.htm
;MenuGetHandle
;https://autohotkey.com/docs/commands/MenuGetHandle.htm
;MenuGetName
;https://autohotkey.com/docs/commands/MenuGetName.htm

;LINKS - GUI (AHK V1)
;GUI
;https://autohotkey.com/docs/commands/Gui.htm
;GuiControl
;https://autohotkey.com/docs/commands/GuiControl.htm
;GuiControlGet
;https://autohotkey.com/docs/commands/GuiControlGet.htm

;LINKS - GUI CONTROLS (AHK V1)
;[includes SB_XXX functions]
;GUI Control Types
;https://autohotkey.com/docs/commands/GuiControls.htm
;[includes IL_XXX functions]
;ListView (GUI)
;https://autohotkey.com/docs/commands/ListView.htm
;TreeView (GUI)
;https://autohotkey.com/docs/commands/TreeView.htm

;==================================================

;GUI/MENU LIBRARY - CLASSES:

;global AHKMenuClass
;global AHKGuiClass
;global AHKCtrlClass
;global AHKMenuClassExists := 1

;==================================================

AHKMenuClassExists()
{
}

;==================================================

;CLASS - MENU:

class AHKMenuClass
{
	static issued := 0
	static varClickCount := 2
	static varStoreHMenu := {} ;AHK v1 Basic Object (combination Object/Map/Array in one)
	__New()
	{
		global
		local Class, hMenu
		Class := RegExReplace(A_ThisFunc, "\..*")
		this.varMenuName := "MenuObj" (1+%Class%.issued++)
		Menu, % this.varMenuName, % "Add"
		Menu, % this.varMenuName, % "DeleteAll"
		hMenu := MenuGetHandle(this.varMenuName)
		%Class%.varStoreHMenu[hMenu+0] := &this
		ObjAddRef(&this)
	}

	ClickCount[]
	{
		get
		{
			if (this.varMenuName != "Tray")
				return
			return this.varClickCount
		}
		set
		{
			if (this.varMenuName != "Tray")
				return
			Menu, % "Tray", % "Click", % value
			return this.varClickCount := value
		}
	}
	Default[]
	{
		get
		{
			if (this.varMenuName != "Tray")
				return
		}
		set
		{
			if (this.varMenuName != "Tray")
				return
			Menu, % "Tray", % "Default", % value
		}
	}
	Handle[]
	{
		get
		{
			return MenuGetHandle(this.varMenuName)
		}
		set
		{
		}
	}

	;MENU ADD: ADD ITEM TO MENU
	Add(Params*) ;MenuItemName, CallbackOrSubmenu, Options
	{
		local CallbackOrSubmenu, MenuItemName, Options
		MenuItemName := Params[1]
		CallbackOrSubmenu := Params[2]
		Options := Params[3]
		;note: the callback function should be of the form:
		;FunctionName(ItemName, ItemPos, MyMenu)
		try
		{
			if !Params.Length()
				Menu, % this.varMenuName, % "Add"
			else if !Params.HasKey(2)
				Menu, % this.varMenuName, % "Add", % MenuItemName,, % Options
			else if IsFunc(CallbackOrSubmenu)
				Menu, % this.varMenuName, % "Add", % MenuItemName, % CallbackOrSubmenu, % Options
			else if IsObject(CallbackOrSubmenu)
				Menu, % this.varMenuName, % "Add", % MenuItemName, % ":" CallbackOrSubmenu.varMenuName, % Options
		}
		catch oError
			throw Exception(oError.Message, -1)
	}
	AddStandard()
	{
		Menu, % "Tray", % "Standard"
	}
	Check(MenuItemName)
	{
		Menu, % this.varMenuName, % "Check", % MenuItemName
	}
	Delete(Params*)
	{
		if Params.HasKey(1)
			Menu, % this.varMenuName, % "Delete", % Params[1]
		else
		{
			Menu, % this.varMenuName, % "NoStandard"
			Menu, % this.varMenuName, % "DeleteAll"
		}
	}
	Disable(MenuItemName)
	{
		Menu, % this.varMenuName, % "Disable", % MenuItemName
	}
	Enable(MenuItemName)
	{
		Menu, % this.varMenuName, % "Enable", % MenuItemName
	}
	Insert(ItemToInsertBefore, NewItemName, CallbackOrSubmenu, Options:="")
	{
		Menu, % this.varMenuName, % "Insert", % ItemToInsertBefore, % NewItemName, % CallbackOrSubmenu, % Options
	}
	Rename(MenuItemName, NewName)
	{
		Menu, % this.varMenuName, % "Rename", % MenuItemName, % NewName
	}
	SetColor(ColorValue, Submenus:=1)
	{
		Menu, % this.varMenuName, % "Color", % ColorValue, % Submenus
	}
	SetIcon(MenuItemName, FileName, IconNumber, IconWidth)
	{
		Menu, % this.varMenuName, % "Icon", % MenuItemName, % FileName, % IconNumber, % IconWidth
	}
	Show(X:="", Y:="")
	{
		Menu, % this.varMenuName, % "Show", % X, % Y
	}
	ToggleCheck(MenuItemName)
	{
		Menu, % this.varMenuName, % "ToggleCheck", % MenuItemName
	}
	ToggleEnable(MenuItemName)
	{
		Menu, % this.varMenuName, % "ToggleEnable", % MenuItemName
	}
	Uncheck(MenuItemName)
	{
		Menu, % this.varMenuName, % "Uncheck", % MenuItemName
	}
}

;==================================================

;CLASS - WINDOW:

class AHKGuiClass
{
	static issued := 0
	static varStoreHWnd := {}
	static varStoreFunc := {}
	__New(Options, Title, EventObj)
	{
		global
		local Class, hWnd

		;for OnEvent (Close/ContextMenu/DropFiles/Escape/Size):
		Options .= " +LabelAHKFCGui_Gui"

		Gui, % "New", % Options " +HwndhWnd", % Title
		this.varHwnd := hWnd
		this.varDPIScale := InStr(Options, "-DPIScale") ? 0 : 1
		this.varEventObj := EventObj

		Class := RegExReplace(A_ThisFunc, "\..*")
		this.varGuiName := "GuiObj" (1+%Class%.issued++)
		;this.Name := this.varGuiName
		this.varDelim := "|"
		this.varNames := {}
		%Class%.varStoreHWnd[hWnd+0] := &this
		ObjAddRef(&this)
	}
	__Call(Method, Params*)
	{
		local ControlType
		;for AddXXX methods:
		if (Method != "Add")
		&& (SubStr(Method, 1, 3) = "Add")
		{
			ControlType := SubStr(Method, 4)
			return this.Add(ControlType, Params[1], Params[2])
		}
	}
	__Get(Property)
	{
		local GuiTempCtrlObj
		if Property in % "BackColor,Control,Ctrl,FocusedCtrl,Hwnd,MarginX,MarginY,MenuBar,Title"
			{} ;noop
		else if Property in % "varBackColor,varData,varDelim,varDPIScale,varGuiName,varHwnd,varIndex,varMarginX,varMarginY,varMax,varNames,varStoreFunc"
			return ObjRawGet(this, Property)
		else if this.varNames.HasKey(Property)
			return this.varNames[Property]
		else
			throw Exception("The specified control/property does not exist.", -1)
	}
	_NewEnum() ;AHK v1-style enumerator
	{
		global ;AHKCtrlClass
		local Addr, CtrlList, DHW, Enum, FuncName, hCtl, Index
		Enum := {}
		Enum.varData := []

		DHW := A_DetectHiddenWindows
		DetectHiddenWindows, % "On"
		WinGet, CtrlList, % "ControlListHwnd", % "ahk_id " this.Hwnd
		Index := 1
		Loop Parse, CtrlList, % "`n"
		{
			hCtl := A_LoopField
			if !AHKCtrlClass.varStoreHCtrl.HasKey(hCtl)
				continue
			Addr := AHKCtrlClass.varStoreHCtrl[hCtl]
			Enum.varData[Index++] := [hCtl, Object(Addr)] ;ObjFromPtrAddRef
		}
		DetectHiddenWindows, % DHW

		Enum.varIndex := 1
		Enum.varMax := Enum.varData.Length()
		FuncName := RegExReplace(A_ThisFunc, "\.[^.]*") ".Next"
		Enum.base := {Next:FuncName}
		return Enum
	}
	Next(ByRef Key, ByRef Value) ;AHK v1-style enumerator
	{
		if (this.varIndex > this.varMax)
			return 0 + 0
		Key := this.varData[this.varIndex, 1]
		Value := this.varData[this.varIndex, 2]
		this.varIndex++
		return 1 + 0
	}
	__Enum() ;AHK v2-style enumerator
	{
		;global
		local Enum, FuncName
		Enum := this._NewEnum()
		FuncName := RegExReplace(A_ThisFunc, "\.[^.]*") ".Next"
		Enum.base := {Call:FuncName}
		return Enum
	}
	Call(ByRef Key, ByRef Value) ;AHK v2-style enumerator
	{
		return this.Next(Key, Value)
	}

	;GUI ADD: ADD CONTROL TO GUI
	Add(ControlType, Options:="", Text:="`f`a`b")
	{
		return new AHKCtrlClass(this, ControlType, Options, Text)
	}
	;Cancel() ;same as Hide [removed from AHK v2]
	;{
	;	Gui, % this.Hwnd ":Cancel"
	;}
	Destroy()
	{
		local hWnd
		if this.varHwnd
		{
			Gui, % this.varHwnd ":Destroy"
			this.varHwnd := 0
		}
		else ;call this.Hwnd to trigger 'Gui has no window.' error message:
			this.Hwnd
	}
	Flash(Blink:=1)
	{
		Gui, % this.Hwnd ":Flash", % Blink ? "" : "Off"
	}
	GetClientPos(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="")
	{
		local Unscale
		AHKFCGui_WinGetClientPos(X, Y, Width, Height, "ahk_id " this.Hwnd)
		Unscale := this.varDPIScale ? (96/A_ScreenDPI) : 1
		Width *= Unscale
		Height *= Unscale

		;possible future functionality:
		;if IsByRef(X) || IsByRef(Y) || IsByRef(Width) || IsByRef(Height)
		;	return ""
		;return [X, Y, Width, Height]
		;return {X:X, Y:Y, W:Width, H:Height}
	}
	GetPos(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="")
	{
		local Unscale
		WinGetPos, X, Y, Width, Height, % "ahk_id " this.Hwnd
		Unscale := this.varDPIScale ? (96/A_ScreenDPI) : 1
		Width *= Unscale
		Height *= Unscale

		;possible future functionality:
		;if IsByRef(X) || IsByRef(Y) || IsByRef(Width) || IsByRef(Height)
		;	return ""
		;return [X, Y, Width, Height]
		;return {X:X, Y:Y, W:Width, H:Height}
	}
	Hide()
	{
		Gui, % this.Hwnd ":Hide"
	}
	Maximize()
	{
		Gui, % this.Hwnd ":Maximize"
	}
	Minimize()
	{
		Gui, % this.Hwnd ":Minimize"
	}
	Move(Params*)
	{
		local Scale
		Scale := this.varDPIScale ? (A_ScreenDPI/96) : 1
		WinMove, % "ahk_id " this.Hwnd,, % Params.HasKey(1) ? Params[1] : "", % Params.HasKey(2) ? Params[2] : "", % Params.HasKey(3) ? Params[3]*Scale : "", % Params.HasKey(4) ? Params[4]*Scale : ""
	}
	OnEvent(EventName, Callback, AddRemove:=1)
	{
		global
		local Callback2, Class, Key2, Key3
		;Print("register event handler:`r`n" EventName " " Callback)
		Class := RegExReplace(A_ThisFunc, "\..*")
		StoreFuncEvent := %Class%.varStoreFunc[this.Hwnd+0, EventName]
		if !StoreFuncEvent
			%Class%.varStoreFunc[this.Hwnd+0, EventName] := StoreFuncEvent := []

		if !IsObject(Callback)
		{
			if IsObject(this.varEventObj)
				Callback := this.varEventObj[Callback].Bind(this)
		}

		Key3 := ""
		for Key2, Callback2 in StoreFuncEvent
		{
			if (Callback2 = Callback)
				Key3 := Key2
		}
		if (AddRemove = 0)
		{
			;in AHK v2, an attempt to delete a callback that doesn't exist, does nothing, and doesn't throw:
			if Key3
				StoreFuncEvent.RemoveAt(Key3)
			return
		}

		;in AHK v2, an attempt to add a callback a second time, does nothing, and doesn't throw:
		if Key3
			return
		if (AddRemove = 1)
			StoreFuncEvent.Push(Callback)
		else if (AddRemove = -1)
			StoreFuncEvent.InsertAt(1, Callback)
	}
	Opt(Options)
	{
		Loop Parse, Options, % " `t"
		{
			if (SubStr(A_LoopField, 1, 10) = "+Delimiter")
				throw Exception("Invalid option.", -2)
		}
		if InStr(Options, "-DPIScale")
			this.varDPIScale := 0
		else if InStr(Options, "+DPIScale")
			this.varDPIScale := 1
		Gui, % this.Hwnd ":" Options
	}
	;Options(Options) ;same as Opt [removed from AHK v2]
	;{
	;	Gui, % this.Hwnd ":" Options
	;}
	Restore()
	{
		Gui, % this.Hwnd ":Restore"
	}
	SetFont(Options, FontName:="")
	{
		Gui, % this.Hwnd ":Font", % Options, % FontName
	}
	Show(Options:="")
	{
		Gui, % this.Hwnd ":Show", % Options
	}
	Submit(Hide:=1)
	{
		;note: it is not necessary to use AHK v1's Gui-Submit subcommand:
		;(we thus avoid the disadvantages of Gui-Submit: it sets variable contents, and makes demands about variable scope:)
		local _, GuiTempCtrlObj, GuiTempMap, GuiTempName
		GuiTempMap := {}
		for GuiTempName, GuiTempCtrlObj in this.varNames
		{
			if (GuiTempCtrlObj.varSubmitIsValue == "")
				continue
			GuiTempMap[GuiTempName] := GuiTempCtrlObj.varSubmitIsValue ? GuiTempCtrlObj.Value : GuiTempCtrlObj.Text

			;note: a ComboBox with AltSubmit returns a position number, else the Edit control text:
			;GUI Control Types - Syntax & Usage | AutoHotkey v2
			;https://www.autohotkey.com/docs/v2/lib/GuiControls.htm#ComboBox
			;MyGui.Submit stores the text, unless the word AltSubmit is in the control's Options and the text matches a list item, in which case it stores the position number of the item.
			if GuiTempCtrlObj.varSubmitIsValue
			&& !GuiTempCtrlObj.Value
			&& (GuiTempCtrlObj.varType = "ComboBox")
				GuiTempMap[GuiTempName] := GuiTempCtrlObj.Text
		}
		return GuiTempMap
	}

	BackColor[]
	{
		get
		{
			return this.varBackColor
		}
		set
		{
			if RegExMatch(value, "i)^(Black|Silver|Gray|White|Maroon|Red|Purple|Fuchsia|Green|Lime|Olive|Yellow|Navy|Blue|Teal|Aqua)$")
			|| (value = "Default")
			|| (value == "")
			|| RegExMatch(value, "i)^0x")
				{} ;noop
			else
				value := Format("0x{:06X}", value)
			Gui, % this.Hwnd ":Color", % value
			this.varBackColor := value
		}
	}
	;ClientPos[]
	;{
	;	get
	;	{
	;		local X, Y, W, H
	;		AHKFCGui_WinGetClientPos(X, Y, W, H, "ahk_id " this.Hwnd)
	;		return {X:X, Y:Y, W:W, H:H}
	;	}
	;	set
	;	{
	;	}
	;}
	;Control[param]
	;{
	;	get
	;	{
	;		;global
	;		;name, ClassNN or HWND
	;		local hWnd, GuiCtrlObj
	;		for hWnd, GuiCtrlObj in this
	;		{
	;			if (GuiCtrlObj.Name = param)
	;			|| (GuiCtrlObj.ClassNN = param)
	;			|| (GuiCtrlObj.Hwnd = param)
	;				return GuiCtrlObj
	;		}
	;
	;		;if (Addr := AHKCtrlClass.varStoreHCtrl[param])
	;		;	return Object(Addr) ;ObjFromPtrAddRef
	;		;ControlGet, hCtl, % "Hwnd",, % param, % "ahk_id " this.Hwnd
	;		;if hCtl
	;		;	return GuiCtrlFromHwnd(hCtl)
	;		;return GuiCtrlFromHwnd(param)
	;	}
	;	set
	;	{
	;	}
	;}
	FocusedCtrl[]
	{
		get
		{
			local ClassNN, DHW, hCtl
			DHW := A_DetectHiddenWindows
			DetectHiddenWindows, % "On"
			ControlGetFocus, ClassNN, % "ahk_id " this.Hwnd
			ControlGet, hCtl, % "Hwnd",, % ClassNN, % "ahk_id " this.Hwnd
			DetectHiddenWindows, % DHW
			return GuiCtrlFromHwnd(hCtl)
		}
		set
		{
		}
	}
	Hwnd[]
	{
		get
		{
			if !this.varHwnd
				throw Exception("Gui has no window.", -2)
			return this.varHwnd
		}
		set
		{
		}
	}
	MarginX[]
	{
		get
		{
			return this.varMarginX
		}
		set
		{
			Gui, % this.Hwnd ":Margin", % value
			this.varMarginX := value
		}
	}
	MarginY[]
	{
		get
		{
			return this.varMarginY
		}
		set
		{
			Gui, % this.Hwnd ":Margin",, % value
			this.varMarginY := value
		}
	}
	;Menu[]
	;{
	;	get
	;	{
	;	}
	;	set
	;	{
	;		Gui, % this.Hwnd ":Menu", % value.varMenuName
	;	}
	;}
	MenuBar[]
	{
		get
		{
		}
		set
		{
			Gui, % this.Hwnd ":Menu", % value.varMenuName
		}
	}
	;Name[] ;Name is just a normal key
	;Pos[]
	;{
	;	get
	;	{
	;		local X, Y, W, H
	;		WinGetPos, X, Y, W, H, % "ahk_id " this.Hwnd
	;		return {X:X, Y:Y, W:W, H:H}
	;	}
	;	set
	;	{
	;	}
	;}
	Title[]
	{
		get
		{
			;note: DllCall avoids using DetectHiddenWindows:
			;WinGetTitle, Title, % "ahk_id " this.Hwnd
			Chars := 1 + DllCall("user32\GetWindowTextLength", "Ptr",this.Hwnd)
			VarSetCapacity(Title, Chars << !!A_IsUnicode)
			DllCall("user32\GetWindowText", "Ptr",this.Hwnd, "Str",Title, "Int",Chars)
			return Title
		}
		set
		{
			;note: DllCall avoids using DetectHiddenWindows:
			;WinSetTitle, % "ahk_id " this.Hwnd,, % value
			DllCall("user32\SetWindowText", "Ptr",this.Hwnd, "Ptr",&value)
		}
	}
}

;==================================================

;CLASS - CONTROL:

class AHKCtrlClass
{
	static issued := 0
	static varStoreHCtrl := {}
	static varStoreFunc := {}
	__New(oGui, ControlType, Options, Text)
	{
		global ;for custom control variables
		local _, Array, Class, Delim, HasDelim, hCtl, hGui, Key, OptionsNew, Value
		hGui := oGui.Hwnd
		Delim := oGui.varDelim
		if (ControlType = "Progress")
			Options := "-Smooth " Options
		else if (ControlType = "ComboBox")
		|| (ControlType = "DropDownList")
		|| (ControlType = "DDL")
		|| (ControlType = "ListBox")
		|| (ControlType = "ListView")
		|| (ControlType = "Tab")
		|| (ControlType = "Tab2")
		|| (ControlType = "Tab3")
		{
			if IsObject(Text)
			{
				Array := Text
				Text := ""
				HasDelim := 0
				for _, Value in Array
				{
					if InStr(Value, Delim) ;case-insensitive
						HasDelim := 1
					Text .= (A_Index == 1) ? Value : Delim Value
				}
				if HasDelim
				{
					Delim := AHKFCGui_StrUnused(1, Text)
					if (Delim = " ")
						Gui, % hGui ":+DelimiterSpace"
					else if (Delim = "`t")
						Gui, % hGui ":+DelimiterTab"
					else
						Gui, % hGui ":+Delimiter" Delim
					Text := ""
					for _, Value in Array
						Text .= (A_Index == 1) ? Value : Delim Value
				}
				else
					Gui, % hGui ":+Delimiter" Delim ;this line shouldn't be necessary, but set delim just in case (e.g. if varDelim wrong, or +Delimiter set)
			}
			else if (Text = "`f`a`b")
				{} ;noop
			else
				throw Exception("Expected an Array but got a String.", -2)
		}
		if !IsObject(Text) && (Text = "`f`a`b")
			Text := ""

		;if (ControlType = "ListBox")
		;&& IsObject(Array2)
		;&& Array2.Length()
		;{
		;	for _, Array3 in Array2
		;	{
		;		;listbox set element n, n=Array3[1], text=Array3[2]
		;	}
		;}

		this.varType := ""
		if (ControlType = "Picture")
		|| (ControlType = "Pic")
		{
			this.varType := "Pic"
			if InStr(Text, ":\")
			&& !FileExist(Text)
				throw Exception("Can't create control.", -2)
		}
		else if (ControlType = "DropDownList")
			this.varType := "DDL"
		else
		{
			for Key, Value in StrSplit("Text,Edit,UpDown,Pic,Button,CheckBox,Radio,DDL,ComboBox,ListBox,ListView,TreeView,Link,Hotkey,DateTime,MonthCal,Slider,Progress,GroupBox,Tab,Tab2,Tab3,StatusBar,ActiveX,Custom", ",")
			{
				if (ControlType = Value)
				{
					this.varType := Value
					break
				}
			}
			if (this.varType == "")
				throw Exception("Invalid control type.", -2)
		}

		;whether .Submit() uses .Value/.Text/neither:
		if RegExMatch(this.varType, "i)^(ActiveX|Button|Custom|GroupBox|Link|ListView|Pic|Progress|StatusBar|Text|TreeView)$")
			this.varSubmitIsValue := ""
		;AHK v2: Use Value instead of Text if AltSubmit was used.
		else if RegExMatch(this.varType, "i)^(ComboBox|DDL|ListBox|Tab|Tab2|Tab3)$")
		{
			;default: Submit() returns .Text for these control types:
			;AltSubmit: Submit() returns .Value (position) for these control types:
			this.varSubmitIsValue := !!InStr(Options, "AltSubmit")
			Options := RegExReplace(Options, "i)\+?AltSubmit") ;note: if we left AltSubmit in the options, that would affect GuiControlGet, causing the Text property backport to return the position
		}
		else
			this.varSubmitIsValue := 1

		oGui.varDelim := Delim

		;Print(AHKFCGui_ControlHwndGetClassNN(hCtl) " " Object(AHKCtrlClass.varStoreHCtrl[hCtl]).ClassNN) ;Object (v1) -> ObjFromPtrAddRef (v2)
		this.varHwndParent := hGui + 0

		Class := RegExReplace(A_ThisFunc, "\..*")
		this.varGuiName := "GuiCtrlObj" (1+%Class%.issued++)
		;we build a new Options string, to remove GUI variable names from the Options parameter,
		;to avoid the 'A control's variable must be global or static.' error:
		OptionsNew := ""
		Loop Parse, Options, % " `t"
		{
			if (SubStr(A_LoopField, 1, 1) = "v")
			&& (A_LoopField != "Vertical")
			&& (A_LoopField != "VScroll")
			{
				this.Name := SubStr(A_LoopField, 2)
				oGui.varNames[this.Name] := this
				continue
			}
			OptionsNew .= " " A_LoopField
		}
		if (ControlType = "ListView")
		|| (ControlType = "TreeView")
			OptionsNew .= " +AltSubmit" ;e.g. for the ItemExpand event

		Gui, % hGui ":Add", % ControlType, % OptionsNew " +HwndhCtl", % Text
		this.varHwnd := hCtl + 0
		%Class%.varStoreHCtrl[hCtl+0] := &this
		ObjAddRef(&this)
	}

	ClassNN[]
	{
		get
		{
			return AHKFCGui_ControlHwndGetClassNN(this.Hwnd)
		}
		set
		{
		}
	}
	Enabled[]
	{
		get
		{
			;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
			ControlGet, IsEnabled, % "Enabled",,, % "ahk_id " this.Hwnd
			return IsEnabled
		}
		set
		{
			;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
			Control, % value ? "Enable" : "Disable",,, % "ahk_id " this.Hwnd
		}
	}
	Focused[]
	{
		get
		{
			local ClassNN, DHW, hCtl
			DHW := A_DetectHiddenWindows
			DetectHiddenWindows, % "On"
			ControlGetFocus, ClassNN, % "ahk_id " this.varHwndParent
			ControlGet, hCtl, % "Hwnd",, % ClassNN, A
			DetectHiddenWindows, % DHW
			return (hCtl = this.Hwnd)
		}
		set
		{
		}
	}
	Gui[]
	{
		get
		{
			return GuiFromHwnd(this.varHwndParent)
		}
		set
		{
		}
	}
	Hwnd[]
	{
		get
		{
			return this.varHwnd
		}
		set
		{
		}
	}
	Name[]
	{
		get
		{
			local GuiTempCtrlObj, GuiTempName, oGui
			oGui := GuiFromHwnd(this.varHwndParent)
			for GuiTempName, GuiTempCtrlObj in oGui.varNames
			{
				if (GuiTempCtrlObj = this)
					return GuiTempName
			}
		}
		set
		{
			local GuiTempCtrlObj, GuiTempName, oGui
			oGui := GuiFromHwnd(this.varHwndParent)
			for GuiTempName, GuiTempCtrlObj in oGui.varNames
			{
				if (GuiTempCtrlObj = this)
				{
					oGui.varNames.Delete(GuiTempName)
					break
				}
			}
			if oGui.varNames.HasKey(value)
				throw Exception("A control with this name already exists.", -1)
			oGui.varNames[value] := this
		}
	}
	;Pos[]
	;{
	;	get
	;	{
	;		local X, Y, W, H
	;		AHKFCGui_ControlGetClientPos(X, Y, W, H, this.ClassNN, "ahk_id " this.varHwndParent)
	;		return {X:X, Y:Y, W:W, H:H}
	;	}
	;	set
	;	{
	;	}
	;}
	Text[]
	{
		;GuiControl Object - Methods & Properties | AutoHotkey v2
		;https://www.autohotkey.com/docs/v2/lib/GuiControl.htm#Text
		;Changes from v1.1 | AutoHotkey v2
		;https://www.autohotkey.com/docs/v2/v2-changes.htm#guicontrolget
		get
		{
			;global
			local Array, CountSel, CtlStyle, Delim, DHW, hCtl, hGui, Text
			hGui := this.varHwndParent
			if (this.Type = "ListBox") ;currently selected item/tab
			{
				GuiControlGet, Text,, % this.Hwnd
				;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
				ControlGet, CtlStyle, % "Style",,, % "ahk_id " this.Hwnd
				if (CtlStyle & 0x8) ;LBS_MULTIPLESEL := 0x8
				{
					SendMessage, 0x190, 0, 0,, % "ahk_id " hCtl ;LB_GETSELCOUNT := 0x190
					CountSel := ErrorLevel
					Array := StrSplit(Text, GuiFromHwnd(hGui).varDelim)
					if (CountSel != Array.Length())
					{
						Delim := AHKFCGui_StrUnused(1, Text)
						if (Delim = " ")
							Gui, % hGui ":+DelimiterSpace"
						else if (Delim = "`t")
							Gui, % hGui ":+DelimiterTab"
						else
							Gui, % hGui ":+Delimiter" Delim
						GuiControlGet, Text,, % this.Hwnd
						GuiFromHwnd(hGui).varDelim := Delim
						Array := StrSplit(Text, Delim)
						if (CountSel != Array.Length())
							throw Exception("listbox get text failed", -2)
					}
					return Array
				}
			}
			else if RegExMatch(this.Type, "i)^(ComboBox|DDL|Tab|Tab2|Tab3)$") ;currently selected item/tab ;(note: no handling for tab controls with TCS_BUTTONS and TCS_MULTISELECT)
			|| RegExMatch(this.Type, "i)^(Pic|Text)$")
				GuiControlGet, Text,, % this.Hwnd
			else if RegExMatch(this.Type, "i)^(DateTime|Edit|Hotkey|MonthCal|Pic|Progress|Slider|UpDown)$")
			{
				DHW := A_DetectHiddenWindows
				DetectHiddenWindows, % "On"
				ControlGetText, Text,, % "ahk_id " this.Hwnd
				DetectHiddenWindows, % DHW
			}
			else ;ActiveX|Button|CheckBox|Custom|GroupBox|Link|ListView|Radio|StatusBar|TreeView
				GuiControlGet, Text,, % this.Hwnd, % "Text"
			return Text
		}
		set
		{
			if RegExMatch(this.Type, "i)^(DateTime|Pic)$")
				return
			else if RegExMatch(this.Type, "i)^(DDL|ListBox|Tab|Tab2|Tab3)$")
				GuiControl, % "ChooseString", % this.Hwnd, % value
			else if RegExMatch(this.Type, "i)^(ActiveX|Edit|Hotkey|ListView|MonthCal|Progress|Slider|TreeView|UpDown)$")
				;note: DllCall avoids using DetectHiddenWindows:
				DllCall("user32\SendMessage", "Ptr",this.Hwnd, "UInt",0xC, "UPtr",0, "Ptr",&value, "Ptr") ;WM_SETTEXT := 0xC
				;DllCall("user32\SetWindowText", "Ptr",this.Hwnd, "Ptr",&value) ;also works
			else if RegExMatch(this.Type, "i)^(Button|Custom|GroupBox|Link|StatusBar|Text)$")
				GuiControl,, % this.Hwnd, % value
			else ;CheckBox|ComboBox|Radio
				GuiControl, % "Text", % this.Hwnd, % value
		}
	}
	Type[]
	{
		get
		{
			return this.varType
		}
		set
		{
		}
	}
	Value[]
	{
		;GuiControl Object - Methods & Properties | AutoHotkey v2
		;https://www.autohotkey.com/docs/v2/lib/GuiControl.htm#Value
		;Changes from v1.1 | AutoHotkey v2
		;https://www.autohotkey.com/docs/v2/v2-changes.htm#guicontrolget
		get
		{
			;global
			local Array, BufInt, CountSel, CtlStyle, DHW, Index, Text
			if RegExMatch(this.Type, "i)^(Button|Custom|GroupBox|Link|ListView|StatusBar|TreeView)$")
				return
			else if (this.Type = "ComboBox")
			|| (this.Type = "DDL")
			{
				if (this.Type = "ComboBox")
				{
					DHW := A_DetectHiddenWindows
					DetectHiddenWindows, % "On"
					ControlGetText, Text, % "Edit1", % "ahk_id " this.Hwnd
					DetectHiddenWindows, % DHW
					if StrLen(Text)
					{
						;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
						ControlGet, Index, % "FindString", % Text,, % "ahk_id " this.Hwnd ;AHK v2: ControlFindItem
						if !ErrorLevel
							return Index
					}
				}

				SendMessage, 0x147, 0, 0,, % "ahk_id " this.Hwnd ;CB_GETCURSEL := 0x147
				Index := ErrorLevel
				if (Index = "FAIL")
					return
				;if (Index == -1) || (Index == 0xFFFFFFFF) ;CB_ERR := -1
				;	return 0 + 0
				;return Index + 1
				return (Index << 32 >> 32) + 1
			}
			else if (this.Type = "ListBox")
			{
				;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
				ControlGet, CtlStyle, % "Style",,, % "ahk_id " this.Hwnd
				if (CtlStyle & 0x808) ;LBS_MULTIPLESEL := 0x8 ;LBS_EXTENDEDSEL := 0x800 ;if has 1 or both styles
				{
					SendMessage, 0x190, 0, 0,, % "ahk_id " this.Hwnd ;LB_GETSELCOUNT := 0x190
					CountSel := ErrorLevel
					if (CountSel == -1) || (CountSel == 0xFFFFFFFF) ;LB_ERR := -1
						throw Exception("get listbox selection failed", -2)
					if !CountSel
						return []
					VarSetCapacity(BufInt, CountSel*4)
					SendMessage, 0x191, % CountSel, % &BufInt,, % "ahk_id " this.Hwnd ;LB_GETSELITEMS := 0x191
					CountSel := ErrorLevel
					if (CountSel == -1) || (CountSel == 0xFFFFFFFF) ;LB_ERR := -1
						throw Exception("get listbox selection failed", -2)
					Array := []
					Loop % CountSel
						Array.Push(NumGet(&BufInt, A_Index*4-4, "Int")+1)
					return Array
				}

				SendMessage, 0x188, 0, 0,, % "ahk_id " this.Hwnd ;LB_GETCURSEL := 0x188
				Index := ErrorLevel
				if (Index == -1) || (Index == 0xFFFFFFFF) ;LB_ERR := -1
					return 0 + 0
				return Index + 1
			}
			else if (this.Type = "Tab")
			|| (this.Type = "Tab2")
			|| (this.Type = "Tab3")
			{
				SendMessage, 0x130B, 0, 0,, % "ahk_id " this.Hwnd ;TCM_GETCURSEL := 0x130B
				Index := ErrorLevel
				if (Index == -1) || (Index == 0xFFFFFFFF)
					return 0 + 0
				return Index + 1
			}
			else ;ActiveX|CheckBox|DateTime|Edit|Hotkey|MonthCal|Pic|Progress|Radio|Slider|Text|UpDown
				GuiControlGet, Text,, % this.Hwnd
			return Text
		}
		set
		{
			;Changes from v1.1 to v2.0
			;https://autohotkey.com/v2/v2-changes.htm
			if RegExMatch(this.Type, "i)^(ActiveX|Button|Custom|GroupBox|Link|ListView|StatusBar|TreeView)$")
				return
			;note: in a multi-select ListBox, will only select 1 item maximum
			else if RegExMatch(this.Type, "i)^(ComboBox|DDL|ListBox|Tab|Tab2|Tab3)$")
			{
				if Value is not % "integer"
					throw Exception("Invalid value.", -2)

				if Value && (this.Type = "ListBox") ;clear any items first, necessary if a multi-select ListBox
					GuiControl, % "Choose", % this.Hwnd, 0

				if !Value && InStr(this.Type, "Tab") ;deselect all tabs in a tab control
				{
					;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
					SendMessage, 0x1330, -1, 0,, % "ahk_id " this.Hwnd ;TCM_SETCURFOCUS := 0x1330
					if (ErrorLevel = "FAIL")
						throw Exception("", -1)
					Sleep, 0
					SendMessage, 0x130C, -1, 0,, % "ahk_id " this.Hwnd ;TCM_SETCURSEL := 0x130C
					if (ErrorLevel = "FAIL")
						throw Exception("", -1)
					return
				}

				GuiControl, % "Choose", % this.Hwnd, % value
			}
			else ;CheckBox|DateTime|Edit|Hotkey|MonthCal|Pic|Progress|Radio|Slider|Text|UpDown
				GuiControl,, % this.Hwnd, % value
		}
	}
	Visible[]
	{
		get
		{
			;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
			ControlGet, IsVisible, % "Visible",,, % "ahk_id " this.varHwnd
			return IsVisible
		}
		set
		{
			;note: ahk_id %ControlHwnd% 'works on hidden controls even when DetectHiddenWindows is Off'
			Control, % value ? "Show" : "Hide",,, % "ahk_id " this.varHwnd
		}
	}

	;CONTROL ADD: ADD ITEM WITHIN CONTROL
	Add(Arg1:="", Params*) ;[ComboBox, DropDownList, ListBox, Tab][ListView, TreeView]
	{
		;global
		local _, Delim, HasDelim, hCtl, hGui, Text, Value
		hGui := this.varHwndParent
		hCtl := this.varHwnd
		if (this.Type = "ListView")
		{
			;Gui, % hGui ":" this.Type, % this.Hwnd ;didn't work?
			Gui, % hGui ":Default"
			Gui, % this.Type, % this.Hwnd
			return LV_Add(Arg1, Params*)
		}
		else if (this.Type = "TreeView")
		{
			Gui, % hGui ":Default"
			Gui, % this.Type, % this.Hwnd
			return TV_Add(Arg1, Params[1], Params[2])
		}

		;AHK v2 only accepts an array, not a string, for the item list:
		if !IsObject(Arg1)
			throw Exception("Parameter #1 of Gui.List.Prototype.Add requires an Array, but received a String.", -2)
		if (this.Type = "ListBox")
		{
			if IsObject(Arg1)
			{
				Loop % Arg1.Length()
					SendMessage, 0x181, -1, % Arg1.GetAddress(A_Index),, % "ahk_id " hCtl ;LB_INSERTSTRING := 0x181
			}
			else
				SendMessage, 0x181, -1, % &Arg1,, % "ahk_id " hCtl ;LB_INSERTSTRING := 0x181
		}
		else if (this.Type = "ComboBox")
		|| (this.Type = "DDL")
		{
			if IsObject(Arg1)
			{
				Loop % Arg1.Length()
					SendMessage, 0x14A, -1, % Arg1.GetAddress(A_Index),, % "ahk_id " hCtl ;CB_INSERTSTRING := 0x14A
			}
			else
				SendMessage, 0x14A, -1, % &Arg1,, % "ahk_id " hCtl ;CB_INSERTSTRING := 0x14A
		}
		else if (this.Type = "Tab")
		|| (this.Type = "Tab2")
		|| (this.Type = "Tab3")
		{
			Delim := GuiFromHwnd(hGui).varDelim
			Text := ""
			HasDelim := 0
			for _, Value in Arg1
			{
				;note: disallow blank strings, because with GuiControl,
				;because concatenating blank strings with the delim char,
				;could result in leading/multiple delim chars,
				;this could cause existing tabs to be replaced,
				;or the tab selection to change:
				if !StrLen(Value)
					throw Exception("tab set text failed", -2)

				if InStr(Value, Delim) ;case-insensitive
					HasDelim := 1
				Text .= (A_Index == 1) ? Value : Delim Value
			}
			if HasDelim
			{
				Delim := AHKFCGui_StrUnused(1, Text)
				if (Delim = " ")
					Gui, % hGui ":+DelimiterSpace"
				else if (Delim = "`t")
					Gui, % hGui ":+DelimiterTab"
				else
					Gui, % hGui ":+Delimiter" Delim
				Text := ""
				for _, Value in Arg1
					Text .= (A_Index == 1) ? Value : Delim Value
				GuiFromHwnd(hGui).varDelim := Delim
			}
			else
				Gui, % hGui ":+Delimiter" Delim ;this line shouldn't be necessary, but set delim just in case (e.g. if varDelim wrong, or +Delimiter set)
			GuiControl,, % this.Hwnd, % Text
		}
	}
	Choose(Value) ;[ComboBox, DropDownList, ListBox, Tab]
	{
		;equivalent to: if (Type(Value) = "Integer")
		if !IsObject(Value)
		&& (ObjGetCapacity([Value], 1) == "")
		&& !InStr(Value, ".")
			GuiControl, % "Choose", % this.Hwnd, % Value
		else
			GuiControl, % "ChooseString", % this.Hwnd, % Value
	}
	Delete(Value:=-1) ;[ComboBox, DropDownList, ListBox, Tab][ListView, TreeView]
	{
		if (this.Type = "ListView")
		{
			Gui, % this.varHwndParent ":Default"
			Gui, % this.Type, % this.Hwnd
			if (Value == -1)
				return LV_Delete()
			else
				return LV_Delete(Value)
		}
		else if (this.Type = "TreeView")
		{
			Gui, % this.varHwndParent ":Default"
			Gui, % this.Type, % this.Hwnd
			if (Value == -1)
				return TV_Delete()
			else
				return TV_Delete(Value)
		}
		else if (this.Type = "ComboBox")
		|| (this.Type = "DDL")
		{
			if (Value == -1)
				SendMessage, 0x14B, 0, 0,, % "ahk_id " this.Hwnd ;CB_RESETCONTENT := 0x14B
			else
				SendMessage, 0x144, % Value-1, 0,, "ahk_id " this.Hwnd ;CB_DELETESTRING := 0x144
		}
		else if (this.Type = "ListBox")
		{
			if (Value == -1)
				SendMessage, 0x184, 0, 0,, "ahk_id " this.Hwnd ;LB_RESETCONTENT := 0x184
			else
				SendMessage, 0x182, % Value-1, 0,, "ahk_id " this.Hwnd ;LB_DELETESTRING := 0x182
		}
		else if (this.Type = "Tab")
		|| (this.Type = "Tab2")
		|| (this.Type = "Tab3")
		{
			if (Value == -1)
				SendMessage, 0x1309, 0, 0,, "ahk_id " this.Hwnd ;TCM_DELETEALLITEMS := 0x1309
			else
				SendMessage, 0x1308, % Value-1, 0,, "ahk_id " this.Hwnd ;TCM_DELETEITEM := 0x1308
		}
	}
	Focus()
	{
		ControlFocus,, % "ahk_id " this.Hwnd
	}
	GetPos(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="")
	{
		local hGui, Unscale
		AHKFCGui_ControlGetClientPos(X, Y, Width, Height, this.ClassNN, "ahk_id " this.varHwndParent)
		hGui := this.varHwndParent
		Unscale := GuiFromHwnd(hGui).varDPIScale ? (96/A_ScreenDPI) : 1
		X *= Unscale
		Y *= Unscale

		;possible future functionality:
		;if IsByRef(X) || IsByRef(Y) || IsByRef(Width) || IsByRef(Height)
		;	return ""
		;return [X, Y, Width, Height]
		;return {X:X, Y:Y, W:Width, H:Height}
	}
	Move(Params*)
	{
		local Pos, Scale
		;hGui := this.varHwndParent
		;Scale := GuiFromHwnd(hGui).varDPIScale ? (A_ScreenDPI/96) : 1
		Scale := 1

		static Letter := StrSplit("xywh")
		Pos := ""
		Loop 4
		{
			if Params.HasKey(A_Index)
			{
				if (A_Index <= 2)
					Pos .= Letter[A_Index] (Params[A_Index]*Scale) " "
				else
					Pos .= Letter[A_Index] Params[A_Index] " "
			}
		}
		GuiControl, % "Move", % this.Hwnd, % Pos
	}
	OnCommand(NotifyCode, Callback, AddRemove:=1)
	{
		local oGui
		static IsReady := 0
		if !IsObject(Callback)
		{
			oGui := GuiFromHwnd(this.varHwndParent)
			if IsObject(oGui.varEventObj)
				Callback := oGui.varEventObj[Callback].Bind(oGui, this)
		}
		if !IsReady
		{
			OnMessage(0x111, "AHKFCGui_OnCommand") ;WM_COMMAND := 0x111
			IsReady := 1
		}
		AHKFCGui_OnCommand(NotifyCode, this, Callback, AddRemove)
	}
	OnEvent(EventName, Callback, AddRemove:=1)
	{
		global
		local Callback2, Class, Key2, Key3
		if !IsObject(Callback)
		{
			oGui := GuiFromHwnd(this.varHwndParent)
			if IsObject(oGui.varEventObj)
				Callback := oGui.varEventObj[Callback].Bind(oGui, this)
		}
		;Print("register event handler:`r`n" EventName " " Callback)
		Class := RegExReplace(A_ThisFunc, "\..*")
		StoreFuncEvent := %Class%.varStoreFunc[this.Hwnd+0, EventName]
		if !StoreFuncEvent
		{
			%Class%.varStoreFunc[this.Hwnd+0, EventName] := StoreFuncEvent := []
			GuiControl, % "+gAHKFCGui_CtrlOnEvent", % this.Hwnd
		}

		Key3 := ""
		for Key2, Callback2 in StoreFuncEvent
		{
			if (Callback2 = Callback)
				Key3 := Key2
		}
		if (AddRemove = 0)
		{
			;in AHK v2, an attempt to delete a callback that doesn't exist, does nothing, and doesn't throw:
			if Key3
				StoreFuncEvent.RemoveAt(Key3)
			return
		}

		;in AHK v2, an attempt to add a callback a second time, does nothing, and doesn't throw:
		if Key3
			return
		if (AddRemove = 1)
			StoreFuncEvent.Push(Callback)
		else if (AddRemove = -1)
			StoreFuncEvent.InsertAt(1, Callback)
	}
	OnNotify(NotifyCode, Callback, AddRemove:=1)
	{
		local oGui
		static IsReady := 0
		if !IsObject(Callback)
		{
			oGui := GuiFromHwnd(this.varHwndParent)
			if IsObject(oGui.varEventObj)
				Callback := oGui.varEventObj[Callback].Bind(oGui, this)
		}
		if !IsReady
		{
			OnMessage(0x4E, "AHKFCGui_OnNotify") ;WM_NOTIFY := 0x4E
			IsReady := 1
		}
		AHKFCGui_OnNotify(NotifyCode, this, Callback, AddRemove)
	}
	Opt(Options)
	{
		GuiControl, % Options, % this.Hwnd
	}
	;Options(Options)
	;{
	;	GuiControl, % Options, % this.Hwnd
	;}
	Redraw()
	{
		GuiControl, % "MoveDraw", % this.Hwnd
	}
	SetFont(Options:="", FontName:="")
	{
		Gui, % this.varHwndParent ":Font", % Options, % FontName
		GuiControl, % "Font", % this.Hwnd
	}
	SetFormat(DateTime)
	{
		GuiControl, % "Text", % this.Hwnd, % DateTime
	}
	UseTab(Value:=0, ExactMatch:=0)
	{
		if !Value
			Gui, % this.varHwndParent ":Tab"
		else if RegExMatch(Value, "^\d+$")
			Gui, % this.varHwndParent ":Tab", % Value
		else if ExactMatch
			Gui, % this.varHwndParent ":Tab", % Value,, % "Exact"
		else
			Gui, % this.varHwndParent ":Tab", % Value
	}

	;StatusBar controls:
	SetIcon(Filename, IconNumber:=1, PartNumber:=1)
	{
		Gui, % this.varHwndParent ":Default"
		SB_SetIcon(Filename, IconNumber, PartNumber)
	}
	SetParts(Width*)
	{
		Gui, % this.varHwndParent ":Default"
		SB_SetParts(Width*)
	}
	SetText(NewText, PartNumber:=1, Style:=0)
	{
		Gui, % this.varHwndParent ":Default"
		SB_SetText(NewText, PartNumber, Style)
	}

	;[GuiControl Object: further (listview/treeview)]
	DeleteCol(ColumnNumber) ;[ListView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return LV_DeleteCol(ColumnNumber)
	}
	Get(ItemID, Attribute) ;[TreeView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return TV_Get(ItemID, Attribute)
	}
	GetChild(ParentItemID) ;[TreeView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return TV_GetChild(ParentItemID)
	}
	GetCount(Mode:="") ;[ListView, TreeView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		if (this.Type = "ListView")
			return LV_GetCount(Mode)
		else if (this.Type = "TreeView")
			return TV_GetCount()
	}
	GetNext(Item, Type) ;[ListView, TreeView]
	{
		;LV.GetNext([StartingRowNumber, RowType])
		;TV.GetNext([ItemID := 0, ItemType := ""])
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		if (this.Type = "ListView")
			return LV_GetNext(Item, Type)
		else if (this.Type = "TreeView")
			return TV_GetNext(Item, Type)
	}
	GetParent(ItemID) ;[TreeView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return TV_GetParent(ItemID)
	}
	GetPrev(ItemID) ;[TreeView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return TV_GetPrev(ItemID)
	}
	GetSelection() ;[TreeView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return TV_GetSelection()
	}
	GetText(Item, ColumnNumber) ;[ListView, TreeView]
	{
		local RetrievedText
		;LV.GetText(RowNumber [, ColumnNumber])
		;TV.GetText(ItemID)
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		if (this.Type = "ListView")
			LV_GetText(RetrievedText, Item, ColumnNumber)
		else if (this.Type = "TreeView")
			TV_GetText(RetrievedText, Item)
		return RetrievedText
	}
	Insert(RowNumber, Options:="", Col*) ;[ListView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return LV_Insert(RowNumber, Options, Col*)
	}
	InsertCol(ColumnNumber, Options:="", ColumnTitle:="") ;[ListView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		return LV_InsertCol(ColumnNumber, Options, ColumnTitle)
	}
	Modify(Item, Options:="", Col*) ;[ListView, TreeView]
	{
		;LV.Modify(RowNumber [, Options, NewCol1, NewCol2, ...])
		;TV.Modify(ItemID [, Options, NewName])
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		if (this.Type = "ListView")
			return LV_Modify(Item, Options, Col*)
		else if (this.Type = "TreeView")
			return TV_Modify(Item, Options, Col*)
	}
	;ModifyCol(ColumnNumber:="", Options:="", ColumnTitle:="") ;[ListView]
	ModifyCol(Params*) ;[ListView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		;return LV_ModifyCol(ColumnNumber, Options, ColumnTitle)
		return LV_ModifyCol(Params*)
	}
	;SetImageList(ImageListID, IconType:="") ;[ListView, TreeView]
	SetImageList(Params*) ;[ListView, TreeView]
	{
		Gui, % this.varHwndParent ":Default"
		Gui, % this.Type, % this.Hwnd
		if (this.Type = "ListView")
			;return LV_SetImageList(ImageListID, IconType) ;IconType: 0 or 1 or 2
			return LV_SetImageList(Params*) ;IconType: 0 or 1 or 2
		else if (this.Type = "TreeView")
			;return TV_SetImageList(ImageListID, IconType) ;IconType: 0 or 2
			return TV_SetImageList(Params*) ;IconType: 0 or 2
	}
}

;==================================================

;LIBRARY - AUXILIARY FUNCTIONS FOR EVENTS (INFO):

;OnEvent (GUI) - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/GuiOnEvent.htm

;Close
;ContextMenu [windows]
;DropFiles
;Escape
;Size

;Change
;Click/DoubleClick/ColClick
;ContextMenu [controls]
;Focus/LoseFocus
;ItemCheck/ItemEdit/ItemExpand/ItemFocus/ItemSelect

;==================================================

;LIBRARY - AUXILIARY FUNCTIONS FOR EVENTS (FUNCTIONS):

;OnEvent (GUI) - Syntax & Usage | AutoHotkey v2
;https://autohotkey.com/docs/v2/lib/GuiOnEvent.htm

;AHKFCGui_GuiOnEvent(oParams*)
;{
;}

AHKFCGui_CtrlOnEvent(hCtl, GuiEvent, EventInfo, ErrLevel:="")
{
	global
	local _, Func, IsRightClick, POINT, Ret, X, Y

	;GUI control events links:
	;[A_GuiEvent/A_GuiControlEvent/A_EventInfo (and A_Gui/A_GuiControl)]
	;[note: ListView/TreeView, Slider/MonthCal, ListBox/StatusBar]
	;Variables and Expressions - Definition & Usage | AutoHotkey
	;https://www.autohotkey.com/docs/v1/Variables.htm#GuiEvent
	;OnEvent (GUI) - Syntax & Usage | AutoHotkey v2
	;https://www.autohotkey.com/docs/v2/lib/GuiOnEvent.htm#Control_Events
	;Changes from v1.1 | AutoHotkey v2
	;https://www.autohotkey.com/docs/v2/v2-changes.htm#events

	;AHK v1: DoubleClick: The event was triggered by a double-click. Note: The first click of the click-pair will still cause a Normal event to be received first. In other words, the subroutine will be launched twice: once for the first click and again for the second.
	if (GuiEvent = "Normal")
	;|| (GuiEvent = "DoubleClick") ;commented out since the first click triggers Normal anyway
	{
		if (GuiCtrlFromHwnd(hCtl).Type = "Edit")
			GuiEvent := "Change"
		else
			GuiEvent := "Click"
	}
	else if (GuiEvent = "RightClick")
		GuiEvent := "ContextMenu"
	else if (GuiEvent == "F") ;case sensitive
		GuiEvent := "Focus"
	else if (GuiEvent == "f") ;case sensitive
		GuiEvent := "LoseFocus"
	else if (GuiEvent = "I")
	{
		;AHK v2: ListView's I event was split into multiple named events, except for the f (de-focus) event, which was excluded because it is implied by F (ItemFocus).
		if InStr(ErrLevel, "c") ;C or c
		{
			Arg3 := !!InStr(ErrLevel, "C", 1) ;case sensitive
			GuiEvent := "ItemCheck"
		}
		else if InStr(ErrLevel, "F", 1) ;case sensitive
			GuiEvent := "ItemFocus"
		else if InStr(ErrLevel, "S") ;S or s
		{
			;warning: ErrLevel (upper case) 'S' (AHK v1) does not appear to work exactly like ItemSelect True (AHK v2):
			Arg3 := !!InStr(ErrLevel, "S", 1) ;case sensitive
			GuiEvent := "ItemSelect"
		}
	}
	else if (GuiEvent = "+") || (GuiEvent = "-")
	{
		Arg3 := (GuiEvent = "+")
		GuiEvent := "ItemExpand"
	}
	else if (GuiEvent == "e") ;case sensitive
		GuiEvent := "ItemEdit"
	else if (GuiEvent = "s")
	&& (GuiCtrlFromHwnd(hCtl).Type = "TreeView")
		GuiEvent := "ItemSelect"
	else if RegExMatch(GuiEvent, "^\d$")
	&& (GuiCtrlFromHwnd(hCtl).Type = "Slider")
	{
		EventInfo := GuiEvent
		GuiEvent := "Change"
	}

	for _, Func in AHKCtrlClass.varStoreFunc[hCtl, GuiEvent]
	{
		;Print("event notification:`r`n" (A_Gui+0) " @ " A_GuiControl " @ " A_GuiEvent " @ " A_GuiControlEvent " @ " A_EventInfo "`r`n`r`n" hCtl " @ " GuiEvent " @ " EventInfo " @ " ErrLevel)

		;if Func is a BoundFunc, we can't know how many implicit leading parameters it has...
		;WARNING: if Func is a BoundFunc, we assume it was given 1 implicit leading parameter:
		;the Type function (AHK v2 and the AHK v1 custom backport) report 'Func'/'BoundFunc' depending on the object:

		;Print(GuiEvent " " EventInfo " " (IsSet(Arg3) ? Arg3 : ""))

		;events: Change
		;events: Click/DoubleClick/ColClick
		;events: Focus/LoseFocus
		;events: ItemEdit/ItemFocus
		;events: ItemSelect (TreeView)
		if RegExMatch(GuiEvent, "i)^(Change|Click|DoubleClick|ColClick|Focus|LoseFocus|ItemEdit|ItemFocus)$")
		|| (GuiEvent = "ItemSelect" && GuiCtrlFromHwnd(hCtl).Type = "TreeView")
			Ret := (Type(Func) = "Func") ? %Func%(GuiCtrlFromHwnd(hCtl), EventInfo) : %Func%(EventInfo)
		;events: ItemCheck/ItemExpand
		;events: ItemSelect (ListView)
		else if RegExMatch(GuiEvent, "i)^(ItemCheck|ItemExpand)$")
		|| (GuiEvent = "ItemSelect" && GuiCtrlFromHwnd(hCtl).Type = "ListView")
			Ret := (Type(Func) = "Func") ? %Func%(GuiCtrlFromHwnd(hCtl), EventInfo, Arg3) : %Func%(EventInfo, Arg3)
		;events: ContextMenu
		;Ctrl_ContextMenu(GuiCtrlObj, Item, IsRightClick, X, Y)
		else if (GuiEvent = "ContextMenu")
		{
			CoordMode, Mouse, Screen
			MouseGetPos, X, Y
			;AHK v2: The ContextMenu and DropFiles events use client coordinates instead of window coordinates (Client is also the default CoordMode in v2).
			;screen to client coordinates:
			VarSetCapacity(POINT, 8, 0)
			NumPut(X, &POINT, 0, "Int")
			NumPut(Y, &POINT, 4, "Int")
			DllCall("user32\ScreenToClient", "Ptr",GuiCtrlFromHwnd(hCtl).varHwndParent, "Ptr",&POINT)
			;Print(X " " Y " " NumGet(&POINT, 0, "Int") " " NumGet(&POINT, 4, "Int"))
			X := NumGet(&POINT, 0, "Int")
			Y := NumGet(&POINT, 4, "Int")
			;warning: assume True for IsRightClick (since it's more common than Shift+10 or AppsKey (Menu key)):
			IsRightClick := True
			Ret := (Type(Func) = "Func") ? %Func%(GuiCtrlFromHwnd(hCtl), EventInfo, IsRightClick, X, Y) : %Func%(EventInfo, IsRightClick, X, Y)
		}
		if (Ret != "")
			break
	}
}

;Ctrl_Change(GuiCtrlObj, Info)
;Ctrl_Click(GuiCtrlObj, Info)
;Ctrl_DoubleClick(GuiCtrlObj, Info)
;Ctrl_ColClick(GuiCtrlObj, Info)
;Ctrl_ContextMenu(GuiCtrlObj, Item, IsRightClick, X, Y)
;Ctrl_Focus(GuiCtrlObj, Info)
;Ctrl_LoseFocus(GuiCtrlObj, Info)
;Ctrl_ItemCheck(GuiCtrlObj, Item, Checked)
;Ctrl_ItemEdit(GuiCtrlObj, Item)
;Ctrl_ItemExpand(GuiCtrlObj, Item, Expanded)
;Ctrl_ItemFocus(GuiCtrlObj, Item)
;ListView_ItemSelect(GuiCtrlObj, Item, Selected)
;TreeView_ItemSelect(GuiCtrlObj, Item)

;Gui - Syntax & Usage | AutoHotkey
;https://www.autohotkey.com/docs/v1/lib/Gui.htm#Labels
AHKFCGui_GuiClose(GuiHwnd)
{
	global
	local _, Func, GuiObj
	GuiObj := Object(AHKGuiClass.varStoreHWnd[GuiHwnd+0]) ;ObjFromPtrAddRef
	for _, Func in AHKGuiClass.varStoreFunc[GuiHwnd, "Close"]
	{
		if (%Func%(GuiObj) != "")
			break
	}
}
AHKFCGui_GuiContextMenu(GuiHwnd, CtrlHwnd, Item, IsRightClick, X, Y)
{
	global
	local _, Func, GuiCtrlObj, GuiObj, POINTC, RECTW

	;AHK v2: The ContextMenu and DropFiles events use client coordinates instead of window coordinates (Client is also the default CoordMode in v2).
	;window to client coordinates:
	VarSetCapacity(RECTW, 16, 0)
	VarSetCapacity(POINTC, 8, 0)
	DllCall("user32\GetWindowRect", "Ptr",GuiHwnd, "Ptr",&RECTW)
	DllCall("user32\ClientToScreen", "Ptr",GuiHwnd, "Ptr",&POINTC)
	;X0 := X, Y0 := Y
	X -= NumGet(&POINTC, 0, "Int") - NumGet(&RECTW, 0, "Int")
	Y -= NumGet(&POINTC, 4, "Int") - NumGet(&RECTW, 4, "Int")
	;Print(X0 " " Y0 " " X " " Y)

	GuiObj := Object(AHKGuiClass.varStoreHWnd[GuiHwnd+0]) ;ObjFromPtrAddRef
	GuiCtrlObj := Object(AHKCtrlClass.varStoreHWnd[CtrlHwnd+0]) ;ObjFromPtrAddRef
	for _, Func in AHKGuiClass.varStoreFunc[GuiHwnd, "ContextMenu"]
	{
		if (%Func%(GuiObj, GuiCtrlObj, Item, IsRightClick, X, Y) != "")
			break
	}
}
AHKFCGui_GuiDropFiles(GuiHwnd, FileArray, CtrlHwnd, X, Y)
{
	global
	local _, Func, GuiCtrlObj, GuiObj, POINTC, RECTW

	;AHK v2: The ContextMenu and DropFiles events use client coordinates instead of window coordinates (Client is also the default CoordMode in v2).
	;window to client coordinates:
	VarSetCapacity(RECTW, 16, 0)
	VarSetCapacity(POINTC, 8, 0)
	DllCall("user32\GetWindowRect", "Ptr",GuiHwnd, "Ptr",&RECTW)
	DllCall("user32\ClientToScreen", "Ptr",GuiHwnd, "Ptr",&POINTC)
	;X0 := X, Y0 := Y
	X -= NumGet(&POINTC, 0, "Int") - NumGet(&RECTW, 0, "Int")
	Y -= NumGet(&POINTC, 4, "Int") - NumGet(&RECTW, 4, "Int")
	;Print(X0 " " Y0 " " X " " Y)

	GuiObj := Object(AHKGuiClass.varStoreHWnd[GuiHwnd+0]) ;ObjFromPtrAddRef
	GuiCtrlObj := Object(AHKCtrlClass.varStoreHWnd[CtrlHwnd+0]) ;ObjFromPtrAddRef
	;note: different order: Gui_DropFiles(GuiObj, GuiCtrlObj, FileArray, X, Y)
	;AHK v2: The DropFiles event swaps the FileArray and Ctrl parameters, to be consistent with ContextMenu.
	for _, Func in AHKGuiClass.varStoreFunc[GuiHwnd, "DropFiles"]
	{
		if (%Func%(GuiObj, GuiCtrlObj, FileArray, X, Y) != "")
			break
	}
}
AHKFCGui_GuiEscape(GuiHwnd)
{
	global
	local _
	;no GuiEscape() example here:
	;GUI
	;https://autohotkey.com/docs/commands/Gui.htm#GuiEscape
	local Func, GuiObj
	GuiObj := Object(AHKGuiClass.varStoreHWnd[GuiHwnd+0]) ;ObjFromPtrAddRef
	for _, Func in AHKGuiClass.varStoreFunc[GuiHwnd, "Escape"]
	{
		if (%Func%(GuiObj) != "")
			break
	}
}
AHKFCGui_GuiSize(GuiHwnd, MinMax, Width, Height)
{
	global
	local _, Func, GuiObj
	GuiObj := Object(AHKGuiClass.varStoreHWnd[GuiHwnd+0]) ;ObjFromPtrAddRef
	;AHK v2: The Size event passes 0, -1 or 1 (consistent with WinGetMinMax) instead of 0, 1 or 2.
	if (MinMax = 1)
		MinMax := -1 ;Min
	else if (MinMax := 2)
		MinMax = 1 ;Max
	for _, Func in AHKGuiClass.varStoreFunc[GuiHwnd, "Size"]
	{
		if (%Func%(GuiObj, MinMax, Width, Height) != "")
			break
	}
}

AHKFCGui_OnCommand(wParam, lParam, uMsg, hWnd)
{
	local _, AddRemove, Callback, Callback2, Code, GuiCtrlObj, hCtl, hWndParent, Key, Key2, Key3, Ret, Sfx
	static StoreFunc := {}
	static StoreCtrl := {}
	if IsObject(lParam)
	{
		Code := wParam
		GuiCtrlObj := lParam
		hCtl := GuiCtrlObj.hWnd
		Callback := uMsg
		AddRemove := hWnd
		Key := (0+hCtl) "_" (0+Code)
		if !StoreFunc.HasKey(Key)
			StoreFunc[Key] := []
		if !StoreCtrl.HasKey(hCtl)
			StoreCtrl[hCtl] := GuiCtrlObj

		Key3 := ""
		for Key2, Callback2 in StoreFunc[Key]
		{
			if (Callback2 = Callback)
				Key3 := Key2
		}
		if (AddRemove = 0)
		{
			;in AHK v2, an attempt to delete a callback that doesn't exist, does nothing, and doesn't throw:
			if Key3
				StoreFunc[Key].RemoveAt(Key3)
			return
		}

		;in AHK v2, an attempt to add a callback a second time, does nothing, and doesn't throw:
		if Key3
			return
		if (AddRemove = 1)
			StoreFunc[Key].Push(Callback)
		else if (AddRemove = -1)
			StoreFunc[Key].InsertAt(1, Callback)
		return
	}

	Code := wParam >> 16 ;high word
	;ID := wParam & 0xFFFF ;low word
	hCtl := lParam
	if !StoreCtrl.HasKey(hCtl+0)
		return ""
	Key := (0+hCtl) "_" (0+Code)
	if StoreFunc.HasKey(Key)
	{
		;AHK v2: If multiple callbacks have been registered for an event, a callback may return a non-empty value to prevent any remaining callbacks from being called.
		;'return' and 'return ""' count as empty
		;'return 0' counts as non-empty
		for _, Callback in StoreFunc[Key]
		{
			Ret := Callback.Call(StoreCtrl[hCtl])
			if (Ret != "")
				break
		}
		return Ret
	}
	return ""
}

AHKFCGui_OnNotify(wParam, lParam, uMsg, hWnd)
{
	local _, AddRemove, Callback, Callback2, Code, GuiCtrlObj, hCtl, hWndParent, Key, Key2, Key3, Ret, Sfx
	static StoreFunc := {}
	static StoreCtrl := {}
	if IsObject(lParam)
	{
		Code := wParam & 0xFFFFFFFF
		GuiCtrlObj := lParam
		hCtl := GuiCtrlObj.hWnd
		Callback := uMsg
		AddRemove := hWnd
		Key := (0+hCtl) "_" (0+Code)
		if !StoreFunc.HasKey(Key)
			StoreFunc[Key] := []
		if !StoreCtrl.HasKey(hCtl)
			StoreCtrl[hCtl] := GuiCtrlObj

		Key3 := ""
		for Key2, Callback2 in StoreFunc[Key]
		{
			if (Callback2 = Callback)
				Key3 := Key2
		}
		if (AddRemove = 0)
		{
			;in AHK v2, an attempt to delete a callback that doesn't exist, does nothing, and doesn't throw:
			if Key3
				StoreFunc[Key].RemoveAt(Key3)
			return
		}

		;in AHK v2, an attempt to add a callback a second time, does nothing, and doesn't throw:
		if Key3
			return
		if (AddRemove = 1)
			StoreFunc[Key].Push(Callback)
		else if (AddRemove = -1)
			StoreFunc[Key].InsertAt(1, Callback)
		return
	}

	hCtl := NumGet(lParam+0, 0, "Ptr") ;hwndFrom
	;ID := NumGet(lParam+0, A_PtrSize=8?8:4, "UPtr") ;idFrom
	Code := NumGet(lParam+0, A_PtrSize=8?16:8, "UInt") ;code
	if !StoreCtrl.HasKey(hCtl+0)
		return ""
	Key := (0+hCtl) "_" (0+Code)
	if StoreFunc.HasKey(Key)
	{
		;AHK v2: If multiple callbacks have been registered for an event, a callback may return a non-empty value to prevent any remaining callbacks from being called.
		;'return' and 'return ""' count as empty
		;'return 0' counts as non-empty
		for _, Callback in StoreFunc[Key]
		{
			Ret := Callback.Call(StoreCtrl[hCtl], lParam)
			if (Ret != "")
				break
		}
		return Ret
	}
	return ""
}

;==================================================

;LIBRARY - FUNCTIONS:

;AHK v2 GUI functions for AHK v1

;Gui [was previously GuiCreate]
;GuiCtrlFromHwnd
;GuiFromHwnd
;MenuBar [was previously MenuBarCreate]
;Menu [was previously MenuCreate]
;MenuFromHandle

global A_TrayMenu

Gui(Options:="", Title:="`f`a`b", EventObj:="")
{
	if (Title = "`f`a`b")
		Title := A_ScriptName
	return new AHKGuiClass(Options, Title, EventObj)
}
GuiCtrlFromHwnd(hWnd)
{
	global
	local Addr
	Addr := AHKCtrlClass.varStoreHCtrl[hWnd+0]
	return Object(Addr) ;ObjFromPtrAddRef
}
GuiFromHwnd(hWnd, RecurseParent:=0)
{
	global
	local Addr
	Addr := AHKGuiClass.varStoreHWnd[hWnd+0]
	if Addr
		return Object(Addr) ;ObjFromPtrAddRef
	if !RecurseParent
		return ""
	Loop
	{
		hWnd := DllCall("user32\GetParent", "Ptr",hWnd, "Ptr")
		Addr := AHKGuiClass.varStoreHWnd[hWnd+0]
		if Addr
			return Object(Addr) ;ObjFromPtrAddRef

		;GWL_STYLE := -16 ;WS_CHILD := 0x40000000
		if !(DllCall("user32\GetWindowLong" (A_PtrSize=8?"Ptr":""), "Ptr",hWnd, "Int",-16, "Ptr") & 0x40000000)
			break
	}
	return ""
}
MenuBar()
{
	global
	local MenuBar
	MenuBar := new AHKMenuClass
	MenuBar.varIsBar := 1
	return MenuBar
}
Menu()
{
	global
	;static Init := MenuCreate()
	;if !Init
	;{
	;	A_TrayMenu := new AHKMenuClass
	;	A_TrayMenu.varMenuName := "Tray"
	;	return 1
	;}
	return new AHKMenuClass
}
MenuFromHandle(hMenu)
{
	global
	local Addr
	Addr := AHKMenuClass.varStoreHMenu[hMenu+0]
	return Object(Addr) ;ObjFromPtrAddRef
}

;==================================================

;LIBRARY - AUXILIARY FUNCTIONS:

AHKFCGui_ControlHwndGetClassNN(hCtl)
{
	local CtlClass, DHW, hCtl2, hWnd
	DHW := A_DetectHiddenWindows
	DetectHiddenWindows, % "On"
	if !hCtl
		throw Exception("Target control not found.", -1)
	WinGetClass, CtlClass, % "ahk_id " hCtl
	if ErrorLevel
		throw Exception("", -1)
	;hWnd := DllCall("user32\GetAncestor", "Ptr",hCtl, "UInt",1, "Ptr") ;GA_PARENT := 1
	;hWnd := DllCall("user32\GetAncestor", "Ptr",hCtl, "UInt",2, "Ptr") ;GA_ROOT := 2
	hWnd := AHKFCGui_GetNonChildParent(hCtl) ;non-child parent
	Loop
	{
		ControlGet, hCtl2, % "Hwnd",, % CtlClass A_Index, % "ahk_id " hWnd
		if !hCtl2
			break
		else if (hCtl == hCtl2)
		{
			DetectHiddenWindows, % DHW
			return CtlClass A_Index
		}
	}
	DetectHiddenWindows, % DHW
}

AHKFCGui_ControlGetClientPos(ByRef X, ByRef Y, ByRef W, ByRef H, Control:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local DHW, hCtl, hWndParent, RECT
	DHW := A_DetectHiddenWindows
	DetectHiddenWindows, % "On"
	ControlGet, hCtl, % "Hwnd",, % Control, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
	DetectHiddenWindows, % DHW
	hWndParent := DllCall("user32\GetAncestor", "Ptr",hCtl, "UInt",1, "Ptr") ;GA_PARENT := 1
	VarSetCapacity(RECT, 16, 0)
	DllCall("user32\GetWindowRect", "Ptr",hCtl, "Ptr",&RECT)
	DllCall("user32\MapWindowPoints", "Ptr",0, "Ptr",hWndParent, "Ptr",&RECT, "UInt",2)
	X := NumGet(&RECT, 0, "Int")
	Y := NumGet(&RECT, 4, "Int")
	W := NumGet(&RECT, 8, "Int") - X
	H := NumGet(&RECT, 12, "Int") - Y
}

AHKFCGui_WinGetClientPos(ByRef X:="", ByRef Y:="", ByRef Width:="", ByRef Height:="", WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local DHW, hWnd, RECT
	DHW := A_DetectHiddenWindows
	DetectHiddenWindows, % "On"
	hWnd := WinExist(WinTitle, WinText, ExcludeTitle, ExcludeText)
	DetectHiddenWindows, % DHW
	VarSetCapacity(RECT, 16, 0)
	DllCall("user32\GetClientRect", "Ptr",hWnd, "Ptr",&RECT)
	DllCall("user32\ClientToScreen", "Ptr",hWnd, "Ptr",&RECT)
	X := NumGet(&RECT, 0, "Int")
	Y := NumGet(&RECT, 4, "Int")
	Width := NumGet(&RECT, 8, "Int")
	Height := NumGet(&RECT, 12, "Int")
}

;note: similar to AHK source code's GetNonChildParent
;note: this should exactly match AHKFC_GetNonChildParent
AHKFCGui_GetNonChildParent(hCtl)
{
	local hWnd, hWndParent, hWndTemp
	hWndTemp := hCtl
	Loop
	{
		;GWL_STYLE := -16 ;WS_CHILD := 0x40000000
		;if not a child window or has no parent
		if !(DllCall("user32\GetWindowLong" (A_PtrSize=8?"Ptr":""), "Ptr",hWndTemp, "Int",-16, "Ptr") & 0x40000000)
		|| !(hWndParent := DllCall("user32\GetParent", "Ptr",hWndTemp, "Ptr"))
		{
			hWnd := hWndTemp
			break
		}
		hWndTemp := hWndParent
	}
	return hWnd
}

;find unused characters not in string(s)
;note: case insensitive: if string contains 'a', then 'a' *and* 'A' are considered used
;vNum: number of chars to return (chars which do not appear in any of the oArray strings)
AHKFCGui_StrUnused(vNum, oParams*)
{
	local vCount, vOutput, vText
	if !oParams.Length()
		throw Exception("No strings specified.", -1)
	VarSetCapacity(vText, 1000*oParams.Length()*2)
	Loop % oParams.Length()
		vText .= oParams[A_Index]
	vCount := 0
	vOutput := ""
	Loop 65535
	{
		if !InStr(vText, Chr(A_Index)) ;case-insensitive,
		{
			vOutput .= Chr(A_Index)
			vCount++
			if (vCount = vNum)
				break
		}
	}
	if (vNum != vCount)
		throw Exception("No unused character available.", -1)
	return vOutput
}

;[FIXME] has Scale been applied correctly in methods?
;oGui.Move(): W H
;oCtl.Move(): X Y
;AHKFCGui_DPIScale(x)
;{
;	;return DllCall("kernel32\MulDiv", "Int",x, "Int",A_ScreenDPI, "Int",96)
;	return x * (A_ScreenDPI/96)
;}

;[FIXME] has Unscale been applied correctly in methods?
;oGui.GetPos(): W H
;oGui.GetClientPos(): W H
;oCtl.GetPos(): X Y
;AHKFCGui_DPIUnscale(x)
;{
;	;return DllCall("kernel32\MulDiv", "Int",x, "Int",96, "Int",A_ScreenDPI)
;	return x * (96/A_ScreenDPI)
;}

;==================================================

;WinActive(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
;{
;	local hWnd := 0
;	IfWinActive, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
;		WinGet, hWnd, ID
;	return hWnd ? hWnd + 0 : 0
;}

;WinExist(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
;{
;	local hWnd
;	WinGet, hWnd, ID, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
;	return hWnd ? hWnd + 0 : 0
;}

WinActive(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd := 0
	if WinTitle is % "integer"
	{
		if (Type(WinTitle) == "Integer")
		{
			IfWinActive, % "ahk_id " WinTitle
				WinGet, hWnd, ID
		}
		else
		{
			IfWinActive, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
				WinGet, hWnd, ID
		}
	}
	else if IsObject(WinTitle)
	{
		if !WinTitle.HasKey("Hwnd")
			throw Exception("", -1)
		IfWinActive, % "ahk_id " WinTitle.Hwnd
			WinGet, hWnd, ID
	}
	else ;note: 'else IfWinActive' caused a (custom) parser warning, so split it into 2 lines:
	{
		IfWinActive, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
			WinGet, hWnd, ID
	}
	return hWnd ? hWnd + 0 : 0
}

WinExist(WinTitle:="", WinText:="", ExcludeTitle:="", ExcludeText:="")
{
	local hWnd
	if WinTitle is % "integer"
	{
		if (Type(WinTitle) == "Integer")
			WinGet, hWnd, ID, % "ahk_id " WinTitle
		else
			WinGet, hWnd, ID, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
	}
	else if IsObject(WinTitle)
	{
		if !WinTitle.HasKey("Hwnd")
			throw Exception("", -1)
		WinGet, hWnd, ID, % "ahk_id " WinTitle.Hwnd
	}
	else
		WinGet, hWnd, ID, % WinTitle, % WinText, % ExcludeTitle, % ExcludeText
	IfWinExist, % "ahk_id " hWnd ;update Last Found Window
		return 0 + hWnd
	return 0 + 0
}

;==================================================
